/*
 * This source code (.java) file is Copyright  2000 Bill Venners. All rights reserved. 
 * This file accompanies the Jini Service UI Draft Specification, written by Bill
 * Venners and published on the World Wide Web at:
 * 
 *     http://www.artima.com/jini/serviceui/DraftSpec.html,
 *
 * This source file may not be copied, modified, or redistributed EXCEPT as allowed
 * by the following statements: From February 7, 2000 through March 6, 2000, you may
 * copy and/or modify these files to test and experiment with the Service UI API,
 * described in the Jini Service UI Draft Specification. Any bug fixes must be given
 * back to Bill Venners. You may not redistribute this file or any binary (such
 * as .class) files generated from this file. You may not distribute modified versions
 * this files or any binary (such as .class) files generated from modified versions of
 * this file. You may not remove this copyright notice. You may not use this file in
 * printed media without the express permission of Bill Venners. And if that weren't
 * enough, you must destroy all copies of this file, and any binary (such as
 * .class) files generated from this file, by March 7, 2000.
 *
 * BILL VENNERS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THIS
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR PARTICULAR PURPOSE, OR NON-INFRINGEMENT.
 * BILL VENNERS SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY A LICENSEE AS A RESULT
 * OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
*/
package net.jini.lookup.ui.attribute;

import java.util.Set;
import java.util.Collection;
import java.util.Iterator;
import java.util.Collections;

/**
 * UI attribute that lists all the types of which a UI factory
 * is an instance. 
*/
public class UIFactoryTypes implements java.io.Serializable {

    private Set typeNames;

    /**
     * Constructs a <CODE>UIFactoryTypes</CODE> attribute using the
     * passed <CODE>Set</CODE>. The <CODE>Set</CODE> can
     * be mutable or immutable, and must contain only
     * <CODE>String</CODE>s. Each <CODE>String</CODE> must
     * represent a fully qualified Java type name.
     * 
     * @param typeNames A <CODE>Set</CODE> of <CODE>String</CODE>
     * objects. Each element must be non-null and an instance
     * of <CODE>java.lang.String</CODE>.
     * 
     * @throws IllegalArgumentException if <CODE>typeNames</CODE> 
     * is <CODE>null</CODE> or any element of <CODE>typeNames</CODE>
     * set is either <CODE>null</CODE>
     * or not an instance of <CODE>java.lang.String</CODE>.
    */
    public UIFactoryTypes(Set typeNames) {
        
        if (typeNames == null) {
            throw new IllegalArgumentException();
        }

        Iterator it = typeNames.iterator();
        while (it.hasNext()) {
            Object o = it.next();
            if ((o == null) || (!(o instanceof String))) {
                throw new IllegalArgumentException();
            }
        }

        this.typeNames = typeNames;
    }

    /*
     * Returns <CODE>true</CODE> if the UI factory (contained in
     * marshalled form in the same <CODE>UIDescriptor</CODE>) is
     * an instance of the type whose fully qualified name is
     * passed in parameter <CODE>fullyQualifiedName</CODE>.
     * Else, returns <CODE>false</CODE>.
     * 
     * @throws IllegalArgumentException if <CODE>fullyQualifiedName</CODE> 
     * is <CODE>null</CODE>
    public boolean isAssignableTo(String fullyQualifiedName) {
        
        if (typeNames == null) {
            throw new IllegalArgumentException();
        }

        return typeNames.contains(fullyQualifiedName);
    }
    */

    /**
     * Returns <CODE>true</CODE> if the UI factory (contained in
     * marshalled form in the same <CODE>UIDescriptor</CODE>) is
     * an instance of the type passed in parameter <CODE>classObj</CODE>.
     * Else, returns <CODE>false</CODE>.
     *
     * <P>
     * This method compares the fully qualified name
     * of the type represented by the passed <CODE>Class</CODE> with
     * the fully qualified names of types appearing in the
     * inheritance hierarchy defining the UI factory's class. If 
     * a string match is found, this method returns <CODE>true</CODE>.
     * If the UI factory is ultimately loaded with a class loader whose
     * parent-delegation chain doesn't include the class loader 
     * that defined the passed class, a type with the same name
     * would be loaded into a different namespace of
     * the virtual machine. If so, the two types would
     * be considered different by the virtual machine, even though
     * they shared the same fully qualified name.
     * 
     * @throws IllegalArgumentException if <CODE>classObj</CODE> 
     * is <CODE>null</CODE>
    */
    public boolean isAssignableTo(Class classObj) {
        
        if (classObj == null) {
            throw new IllegalArgumentException();
        }

        return typeNames.contains(classObj.getName());
    }

    /*
     * Returns <CODE>true</CODE> if the UI factory (contained in
     * marshalled form in the same <CODE>UIDescriptor</CODE>) is
     * an instance of each type whose fully qualified name appears
     * passed in parameter <CODE>fullyQualifiedNames</CODE>.
     * Else, returns <CODE>false</CODE>.
     * 
     * @throws IllegalArgumentException if <CODE>fullyQualifiedNames</CODE> 
     * is <CODE>null</CODE>, or if any element in the <CODE>Set</CODE> 
     * referenced from <CODE>fullyQualifiedNames</CODE> is <CODE>null</CODE>
     * or is not a <CODE>String</CODE>.
    public boolean isAssignableTo(Set fullyQualifiedNames) {
        
        if (fullyQualifiedNames == null) {
            throw new IllegalArgumentException();
        }

        Iterator it = fullyQualifiedNames.iterator();
        while (it.hasNext()) {
            Object o = it.next();
            if ((o == null) || (!(o instanceof String))) {
                throw new IllegalArgumentException();
            }
        }

        return typeNames.containsAll(fullyQualifiedNames);
    }
    */

    /**
     * Returns an iterator over the set of types of which
     * a UI factory object is an instance in no particular order.
     * The returned <CODE>Iterator</CODE> does not support
     * <CODE>remove()</CODE>.
    */
    public Iterator iterator() {

        // Make sure the iterator that gets passed back
        // doesn't allow client to change the Set.
        return new Iterator() {

            Iterator it = typeNames.iterator();

            public boolean hasNext() {
                return it.hasNext();
            }

            public Object next() {
                return it.next();
            }

            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }

    /**
     * Returns an unmodifiable Set of fully qualified type name
     * <CODE>String</CODE>s of which
     * a UI factory is an instance in no particular order.
     * The returned <CODE>Iterator</CODE> does not support
     * <CODE>remove()</CODE>.
    */
    public Set getTypeNames() {

        return Collections.unmodifiableSet(typeNames);
    }
}
 
