/******************************************************************
*
*	CyberLink for Java
*
*	Copyright (C) Satoshi Konno 2002-2004
*
*	File: Device.java
*
*	Revision;
*
*	11/28/02
*		- first revision.
*	02/26/03
*		- URLBase is updated automatically.
* 		- Description of a root device is returned from the XML node tree.
*	05/13/03
*		- URLBase is updated when the request is received.
*		- Changed to create socket threads each local interfaces.
*		  (HTTP, SSDPSearch)
*	06/17/03
*		- Added notify all state variables when a new subscription is received.
*	06/18/03
*		- Fixed a announce bug when the bind address is null on J2SE v 1.4.1_02 and Redhat 9.
*	09/02/03
*		- Giordano Sassaroli <sassarol@cefriel.it>
*		- Problem : bad request response sent even with successful subscriptions
*		- Error : a return statement is missing in the httpRequestRecieved method
*	10/21/03
*		- Updated a udn field by a original uuid.
*	10/22/03
*		- Added setActionListener().
*		- Added setQueryListener().
*	12/12/03
*		- Added a static() to initialize UPnP class.
*	12/25/03
*		- Added advertiser functions.
*	01/05/04
*		- Added isExpired().
*	
******************************************************************/

package org.cybergarage.upnp;

import java.net.*;
import java.io.*;
import java.util.*;

import org.cybergarage.http.*;
import org.cybergarage.util.*;
import org.cybergarage.xml.*;
import org.cybergarage.soap.*;

import org.cybergarage.upnp.ssdp.*;
import org.cybergarage.upnp.device.*;
import org.cybergarage.upnp.control.*;
import org.cybergarage.upnp.event.*;
import org.cybergarage.upnp.xml.*;

public class Device implements org.cybergarage.http.HTTPRequestListener, SearchListener
{
	////////////////////////////////////////////////
	//	Constants
	////////////////////////////////////////////////
	
	public final static String ELEM_NAME = "device";
	public final static String UPNP_ROOTDEVICE = "upnp:rootdevice";

	public final static int DEFAULT_STARTUP_WAIT_TIME = 1000;
	public final static int DEFAULT_DISCOVERY_WAIT_TIME = 300;
	public final static int DEFAULT_LEASE_TIME = 30 * 60;

	public final static int HTTP_DEFAULT_PORT = 4004;

	public final static String DEFAULT_DESCRIPTION_URI = "/description.xml";
	
	////////////////////////////////////////////////
	//	Member
	////////////////////////////////////////////////

	private Node rootNode;
	private Node deviceNode;

	public Node getRootNode()
	{
		return rootNode;
	}

	public Node getDeviceNode()
	{
		return deviceNode;
	}

	public void setRootNode(Node node)
	{
		rootNode = node;
	}

	public void setDeviceNode(Node node)
	{
		deviceNode = node;
	}
				
	////////////////////////////////////////////////
	//	Initialize
	////////////////////////////////////////////////
	
	static 
	{
		UPnP.initialize();
	}
	
	////////////////////////////////////////////////
	//	Constructor
	////////////////////////////////////////////////

	public Device(Node root, Node device)
	{
		rootNode = root;
		deviceNode = device;
		setUUID(UPnP.createUUID());
	}

	public Device()
	{
		this(null, null);
	}
	
	public Device(Node device)
	{
		this(null, device);
	}

	public Device(File descriptionFile) throws InvalidDescriptionException
	{
		this(null, null);
		loadDescription(descriptionFile);
	}

	public Device(String descriptionFileName) throws InvalidDescriptionException
	{
		this(new File(descriptionFileName));
	}

	////////////////////////////////////////////////
	//	Device UUID
	////////////////////////////////////////////////

	private String devUUID;
	
	private void setUUID(String uuid)
	{
		devUUID = uuid;
	}
	
	private String getUUID() 
	{
		return devUUID;
	}
	
	private void updateUDN()
	{
		setUDN("uuid:" + getUUID());	
	}
	
	////////////////////////////////////////////////
	//	Root Device
	////////////////////////////////////////////////
	
	public Device getRootDevice()
	{
		Node rootNode = getRootNode();
		if (rootNode == null)
			return null;
		Node devNode = rootNode.getNode(Device.ELEM_NAME);
		if (devNode == null)
			return null;
		return new Device(rootNode, devNode);
	}

	////////////////////////////////////////////////
	//	UserData
	////////////////////////////////////////////////

	private DeviceData getDeviceData()
	{
		Node node = getDeviceNode();
		DeviceData userData = (DeviceData)node.getUserData();
		if (userData == null) {
			userData = new DeviceData();
			node.setUserData(userData);
			userData.setNode(node);
		}
		return userData;
	}
	
	////////////////////////////////////////////////
	//	Description
	////////////////////////////////////////////////

	private void setDescriptionFile(File file)
	{
		getDeviceData().setDescriptionFile(file);
	}

	public File getDescriptionFile()
	{
		return getDeviceData().getDescriptionFile();
	}

	private void setDescriptionURI(String uri)
	{
		getDeviceData().setDescriptionURI(uri);
	}

	private String getDescriptionURI()
	{
		return getDeviceData().getDescriptionURI();
	}

	private boolean isDescriptionURI(String uri)
	{
		String descriptionURI = getDescriptionURI();
		if (uri == null || descriptionURI == null)
			return false;
		return descriptionURI.equals(uri);
	}

	public String getDescriptionFilePath()
	{
		File descriptionFile = getDescriptionFile();
		if (descriptionFile == null)
			return "";
		return descriptionFile.getAbsoluteFile().getParent();
	}

	public boolean loadDescription(File file) throws InvalidDescriptionException
	{
		try {
			Parser parser = UPnP.getXMLParser();
			rootNode = parser.parse(file);
			if (rootNode == null)
				throw new InvalidDescriptionException(Description.NOROOT_EXCEPTION, file);
			deviceNode = rootNode.getNode(Device.ELEM_NAME);
			if (deviceNode == null)
				throw new InvalidDescriptionException(Description.NOROOTDEVICE_EXCEPTION, file);
		}
		catch (ParserException e) {
			throw new InvalidDescriptionException(e);
		}
		
		setDescriptionFile(file);
		setDescriptionURI(DEFAULT_DESCRIPTION_URI);
		setLeaseTime(DEFAULT_LEASE_TIME);
		setHTTPPort(HTTP_DEFAULT_PORT);

		updateUDN();
				
		return true;
	}

	////////////////////////////////////////////////
	//	isDeviceNode
	////////////////////////////////////////////////

	public static boolean isDeviceNode(Node node)
	{
		return Device.ELEM_NAME.equals(node.getName());
	}
	
	////////////////////////////////////////////////
	//	Root Device
	////////////////////////////////////////////////

	public boolean isRootDevice()
	{
		return (getRootNode() != null) ? true : false;
	}
	
	////////////////////////////////////////////////
	//	Root Device
	////////////////////////////////////////////////

	public void setSSDPPacket(SSDPPacket packet)
	{
		getDeviceData().setSSDPPacket(packet);
	}

	public SSDPPacket getSSDPPacket()
	{
		if (isRootDevice() == false)
			return null;
		return getDeviceData().getSSDPPacket();
	}
	
	////////////////////////////////////////////////
	//	Location 
	////////////////////////////////////////////////

	public void setLocation(String value)
	{
		getDeviceData().setLocation(value);
	}

	public String getLocation()
	{
		SSDPPacket packet = getSSDPPacket();
		if (packet != null)
			return packet.getLocation();		
		return getDeviceData().getLocation();
	}

	////////////////////////////////////////////////
	//	LeaseTime 
	////////////////////////////////////////////////

	public void setLeaseTime(int value)
	{
		getDeviceData().setLeaseTime(value);
		Advertiser adv = getAdvertiser();
		if (adv != null) {
			announce();
			adv.restart();
		}
	}

	public int getLeaseTime()
	{
		SSDPPacket packet = getSSDPPacket();
		if (packet != null)
			return packet.getLeaseTime();	
		return getDeviceData().getLeaseTime();
	}

	////////////////////////////////////////////////
	//	TimeStamp 
	////////////////////////////////////////////////

	public long getTimeStamp()
	{
		SSDPPacket packet = getSSDPPacket();
		if (packet != null)
			return packet.getTimeStamp();		
		return 0;
	}

	public long getElapsedTime()
	{
		return (System.currentTimeMillis() - getTimeStamp()) / 1000;
	}

	public boolean isExpired()
	{
		long elipsedTime = getElapsedTime();
		long leaseTime = getLeaseTime() + UPnP.DEFAULT_EXPIRED_DEVICE_EXTRA_TIME;
		if (leaseTime < elipsedTime)
			return true;
		return false;
	}
	
	////////////////////////////////////////////////
	//	URL Base
	////////////////////////////////////////////////

	private final static String URLBASE_NAME = "URLBase";
	
	private void setURLBase(String value)
	{
		if (isRootDevice() == true) {
			Node node = getRootNode().getNode(URLBASE_NAME);
			if (node != null) {
				node.setValue(value);
				return;
			}
			node = new Node(URLBASE_NAME);
			node.setValue(value);
			int index = 1;
			if (getRootNode().hasNodes() == false)
				index = 1;
			getRootNode().insertNodeAt(node, index);
		}
	}

	private void updateURLBase(String host)
	{
		String urlBase = HostInterface.getHostURL(host, getHTTPPort(), "");
		setURLBase(urlBase);
	}
  
	public String getURLBase()
	{
		if (isRootDevice() == true)
			return getRootNode().getNodeValue(URLBASE_NAME);
		return "";
	}

	////////////////////////////////////////////////
	//	deviceType
	////////////////////////////////////////////////

	private final static String DEVICE_TYPE = "deviceType";
	
	public void setDeviceType(String value)
	{
		getDeviceNode().setNode(DEVICE_TYPE, value);
	}

	public String getDeviceType()
	{
		return getDeviceNode().getNodeValue(DEVICE_TYPE);
	}

	////////////////////////////////////////////////
	//	friendlyName
	////////////////////////////////////////////////

	private final static String FRIENDLY_NAME = "friendlyName";
	
	public void setFriendlyName(String value)
	{
		getDeviceNode().setNode(FRIENDLY_NAME, value);
	}

	public String getFriendlyName()
	{
		return getDeviceNode().getNodeValue(FRIENDLY_NAME);
	}

	////////////////////////////////////////////////
	//	manufacture
	////////////////////////////////////////////////

	private final static String MANUFACTURE = "manufacture";
	
	public void setManufacture(String value)
	{
		getDeviceNode().setNode(MANUFACTURE, value);
	}

	public String getManufacture()
	{
		return getDeviceNode().getNodeValue(MANUFACTURE);
	}

	////////////////////////////////////////////////
	//	manufactureURL
	////////////////////////////////////////////////

	private final static String MANUFACTURE_URL = "manufactureURL";
	
	public void setManufactureURL(String value)
	{
		getDeviceNode().setNode(MANUFACTURE_URL, value);
	}

	public String getManufactureURL()
	{
		return getDeviceNode().getNodeValue(MANUFACTURE_URL);
	}

	////////////////////////////////////////////////
	//	modelDescription
	////////////////////////////////////////////////

	private final static String MODEL_DESCRIPTION = "modelDescription";
	
	public void setModelDescription(String value)
	{
		getDeviceNode().setNode(MODEL_DESCRIPTION, value);
	}

	public String getModelDescription()
	{
		return getDeviceNode().getNodeValue(MODEL_DESCRIPTION);
	}

	////////////////////////////////////////////////
	//	modelName
	////////////////////////////////////////////////

	private final static String MODEL_NAME = "modelName";
	
	public void setModelName(String value)
	{
		getDeviceNode().setNode(MODEL_NAME, value);
	}

	public String getModelName()
	{
		return getDeviceNode().getNodeValue(MODEL_NAME);
	}

	////////////////////////////////////////////////
	//	modelNumber
	////////////////////////////////////////////////

	private final static String MODEL_NUMBER = "modelNumber";
	
	public void setModelNumber(String value)
	{
		getDeviceNode().setNode(MODEL_NUMBER, value);
	}

	public String getModelNumber()
	{
		return getDeviceNode().getNodeValue(MODEL_NUMBER);
	}

	////////////////////////////////////////////////
	//	modelURL
	////////////////////////////////////////////////

	private final static String MODEL_URL = "modelURL";
	
	public void setModelURL(String value)
	{
		getDeviceNode().setNode(MODEL_URL, value);
	}

	public String getModelURL()
	{
		return getDeviceNode().getNodeValue(MODEL_URL);
	}

	////////////////////////////////////////////////
	//	serialNumber
	////////////////////////////////////////////////

	private final static String SERIAL_NUMBER = "serialNumber";
	
	public void setSerialNumber(String value)
	{
		getDeviceNode().setNode(SERIAL_NUMBER, value);
	}

	public String getSerialNumber()
	{
		return getDeviceNode().getNodeValue(SERIAL_NUMBER);
	}

	////////////////////////////////////////////////
	//	UDN
	////////////////////////////////////////////////

	private final static String UDN = "UDN";
	
	public void setUDN(String value)
	{
		getDeviceNode().setNode(UDN, value);
	}

	public String getUDN()
	{
		return getDeviceNode().getNodeValue(UDN);
	}

	////////////////////////////////////////////////
	//	UPC
	////////////////////////////////////////////////

	private final static String UPC = "UPC";
	
	public void setUPC(String value)
	{
		getDeviceNode().setNode(UPC, value);
	}

	public String getUPC()
	{
		return getDeviceNode().getNodeValue(UPC);
	}

	////////////////////////////////////////////////
	//	presentationURL
	////////////////////////////////////////////////

	private final static String presentationURL = "presentationURL";
	
	public void setPresentationURL(String value)
	{
		getDeviceNode().setNode(presentationURL, value);
	}

	public String getPresentationURL()
	{
		return getURLBase() + getDeviceNode().getNodeValue(presentationURL);
	}

	////////////////////////////////////////////////
	//	deviceList
	////////////////////////////////////////////////

	public DeviceList getDeviceList()
	{
		DeviceList devList = new DeviceList();
		Node devListNode = getDeviceNode().getNode(DeviceList.ELEM_NAME);
		if (devListNode == null)
			return devList;
		int nNode = devListNode.getNNodes();
		for (int n=0; n<nNode; n++) {
			Node node = devListNode.getNode(n);
			if (Device.isDeviceNode(node) == false)
				continue;
			Device dev = new Device(node);
			devList.add(dev);
		} 
		return devList;
	}

	public boolean isDevice(String name)
	{
		if (name == null)
			return false;
		if (name.endsWith(getUDN()) == true)
			return true;
		if (name.equals(getFriendlyName()) == true)
			return true;
		if (name.endsWith(getDeviceType()) == true)
			return true;
		return false;
	}
	
	public Device getDevice(String name)
	{
		DeviceList devList = getDeviceList();
		int devCnt = devList.size();
		for (int n=0; n<devCnt; n++) {
			Device dev = devList.getDevice(n);
			if (dev.isDevice(name) == true)
				return dev;
			Device cdev = dev.getDevice(name);
			if (cdev != null)
				return cdev;
		}
		return null;
	}
	
	////////////////////////////////////////////////
	//	serviceList
	////////////////////////////////////////////////

	public ServiceList getServiceList()
	{
		ServiceList serviceList = new ServiceList();
		Node serviceListNode = getDeviceNode().getNode(ServiceList.ELEM_NAME);
		if (serviceListNode == null)
			return serviceList;
		int nNode = serviceListNode.getNNodes();
		for (int n=0; n<nNode; n++) {
			Node node = serviceListNode.getNode(n);
			if (Service.isServiceNode(node) == false)
				continue;
			Service service = new Service(node);
			serviceList.add(service);
		} 
		return serviceList;
	}

	public Service getService(String name)
	{
		ServiceList serviceList = getServiceList();
		int serviceCnt = serviceList.size();
		for (int n=0; n<serviceCnt; n++) {
			Service service = serviceList.getService(n);
			if (service.isService(name) == true)
				return service;
		}
		
		DeviceList devList = getDeviceList();
		int devCnt = devList.size();
		for (int n=0; n<devCnt; n++) {
			Device dev = devList.getDevice(n);
			Service service = dev.getService(name);
			if (service != null)
				return service;
		}
		
		return null;
	}

	private Service getServiceByControlURL(String searchUrl)
	{
		ServiceList serviceList = getServiceList();
		int serviceCnt = serviceList.size();
		for (int n=0; n<serviceCnt; n++) {
			Service service = serviceList.getService(n);
			if (service.isControlURL(searchUrl) == true)
				return service;
		}
		
		DeviceList devList = getDeviceList();
		int devCnt = devList.size();
		for (int n=0; n<devCnt; n++) {
			Device dev = devList.getDevice(n);
			Service service = dev.getServiceByControlURL(searchUrl);
			if (service != null)
				return service;
		}
		
		return null;
	}

	private Service getServiceByEventSubURL(String searchUrl)
	{
		ServiceList serviceList = getServiceList();
		int serviceCnt = serviceList.size();
		for (int n=0; n<serviceCnt; n++) {
			Service service = serviceList.getService(n);
			if (service.isEventSubURL(searchUrl) == true)
				return service;
		}
		
		DeviceList devList = getDeviceList();
		int devCnt = devList.size();
		for (int n=0; n<devCnt; n++) {
			Device dev = devList.getDevice(n);
			Service service = dev.getServiceByEventSubURL(searchUrl);
			if (service != null)
				return service;
		}
		
		return null;
	}

	public Service getSubscriberService(String uuid)
	{
		ServiceList serviceList = getServiceList();
		int serviceCnt = serviceList.size();
		for (int n=0; n<serviceCnt; n++) {
			Service service = serviceList.getService(n);
			String sid = service.getSID();
			if (uuid.equals(sid) == true)
				return service;
		}
		
		DeviceList devList = getDeviceList();
		int devCnt = devList.size();
		for (int n=0; n<devCnt; n++) {
			Device dev = devList.getDevice(n);
			Service service = dev.getSubscriberService(uuid);
			if (service != null)
				return service;
		}
		
		return null;
	}

	////////////////////////////////////////////////
	//	StateVariable
	////////////////////////////////////////////////

	public StateVariable getStateVariable(String name)
	{
		ServiceList serviceList = getServiceList();
		int serviceCnt = serviceList.size();
		for (int n=0; n<serviceCnt; n++) {
			Service service = serviceList.getService(n);
			StateVariable stateVar = service.getStateVariable(name);
			if (stateVar != null)
				return stateVar;
		}
		
		DeviceList devList = getDeviceList();
		int devCnt = devList.size();
		for (int n=0; n<devCnt; n++) {
			Device dev = devList.getDevice(n);
			StateVariable stateVar = dev.getStateVariable(name);
			if (stateVar != null)
				return stateVar;
		}
		
		return null;
	}

	////////////////////////////////////////////////
	//	Action
	////////////////////////////////////////////////

	public Action getAction(String name)
	{
		ServiceList serviceList = getServiceList();
		int serviceCnt = serviceList.size();
		for (int n=0; n<serviceCnt; n++) {
			Service service = serviceList.getService(n);
			ActionList actionList = service.getActionList();
			int actionCnt = actionList.size();
			for (int i=0; i<actionCnt; i++) {
				Action action = (Action)actionList.getAction(i);
				String actionName = action.getName();
				if (actionName == null)
					continue;
				if (actionName.equals(name) == true)
					return action;
			}
		}
		
		DeviceList devList = getDeviceList();
		int devCnt = devList.size();
		for (int n=0; n<devCnt; n++) {
			Device dev = devList.getDevice(n);
			Action action = dev.getAction(name);
			if (action != null)
				return action;
		}
		
		return null;
	}

	////////////////////////////////////////////////
	//	iconList
	////////////////////////////////////////////////

	public IconList getIconList()
	{
		IconList iconList = new IconList();
		Node iconListNode = getDeviceNode().getNode(IconList.ELEM_NAME);
		if (iconListNode == null)
			return iconList;
		int nNode = iconListNode.getNNodes();
		for (int n=0; n<nNode; n++) {
			Node node = iconListNode.getNode(n);
			if (Icon.isIconNode(node) == false)
				continue;
			Icon icon = new Icon(node);
			iconList.add(icon);
		} 
		return iconList;
	}
	
	public Icon getIcon(int n)
	{
		IconList iconList = getIconList();
		if (n < 0 && (iconList.size()-1) < n)
			return null;
		return iconList.getIcon(n);
	}

	////////////////////////////////////////////////
	//	Notify
	////////////////////////////////////////////////

	private String getLocationURL(String host)
	{
		return HostInterface.getHostURL(host, getHTTPPort(), getDescriptionURI());
	}

	private String getNotifyDeviceNT()
	{
		if (isRootDevice() == false)
			return getUDN();			
		return UPNP_ROOTDEVICE;
	}

	private String getNotifyDeviceUSN()
	{
		if (isRootDevice() == false)
			return getUDN();			
		return getUDN() + "::" + UPNP_ROOTDEVICE;
	}

	private String getNotifyDeviceTypeNT()
	{
		return getDeviceType();
	}

	private String getNotifyDeviceTypeUSN()
	{
		return getUDN() + "::" + getDeviceType();
	}
	
	public final static void notifyWait()
	{
		TimerUtil.waitRandom(DEFAULT_DISCOVERY_WAIT_TIME);
	}
		
	public void announce(String bindAddr)
	{
		String devLocation = getLocationURL(bindAddr);
		
		SSDPNotifySocket ssdpSock = new SSDPNotifySocket(bindAddr);

		SSDPNotifyRequest ssdpReq = new SSDPNotifyRequest();
		ssdpReq.setServer(UPnP.getServerName());
		ssdpReq.setLeaseTime(getLeaseTime());
		ssdpReq.setLocation(devLocation);
		ssdpReq.setNTS(NTS.ALIVE);
		
		// uuid:device-UUID(::upnp:rootdevice)* 
		String devNT = getNotifyDeviceNT();			
		String devUSN = getNotifyDeviceUSN();
		ssdpReq.setNT(devNT);
		ssdpReq.setUSN(devUSN);
		notifyWait();
		ssdpSock.post(ssdpReq);

		// uuid:device-UUID::urn:schemas-upnp-org:device:deviceType:v 
		devNT = getNotifyDeviceTypeNT();			
		devUSN = getNotifyDeviceTypeUSN();
		ssdpReq.setNT(devNT);
		ssdpReq.setUSN(devUSN);
		notifyWait();
		ssdpSock.post(ssdpReq);
		
		ServiceList serviceList = getServiceList();
		int serviceCnt = serviceList.size();
		for (int n=0; n<serviceCnt; n++) {
			Service service = serviceList.getService(n);
			service.announce(bindAddr);
		}

		DeviceList childDeviceList = getDeviceList();
		int childDeviceCnt = childDeviceList.size();
		for (int n=0; n<childDeviceCnt; n++) {
			Device childDevice = childDeviceList.getDevice(n);
			childDevice.announce(bindAddr);
		}
	}

	public void announce()
	{
		int nHostAddrs = HostInterface.getNHostAddresses();
		for (int n=0; n<nHostAddrs; n++) {
			String bindAddr = HostInterface.getHostAddress(n);
			if (bindAddr == null || bindAddr.length() <= 0)
				continue;
			announce(bindAddr);
		}
	}
	
	public void byebye(String bindAddr)
	{
		SSDPNotifySocket ssdpSock = new SSDPNotifySocket(bindAddr);
		
		SSDPNotifyRequest ssdpReq = new SSDPNotifyRequest();
		ssdpReq.setNTS(NTS.BYEBYE);
		
		// uuid:device-UUID(::upnp:rootdevice)* 
		String devNT = getNotifyDeviceNT();			
		String devUSN = getNotifyDeviceUSN();
		ssdpReq.setNT(devNT);
		ssdpReq.setUSN(devUSN);
		notifyWait();
		ssdpSock.post(ssdpReq);

		// uuid:device-UUID::urn:schemas-upnp-org:device:deviceType:v 
		devNT = getNotifyDeviceTypeNT();			
		devUSN = getNotifyDeviceTypeUSN();
		ssdpReq.setNT(devNT);
		ssdpReq.setUSN(devUSN);
		notifyWait();
		ssdpSock.post(ssdpReq);

		ServiceList serviceList = getServiceList();
		int serviceCnt = serviceList.size();
		for (int n=0; n<serviceCnt; n++) {
			Service service = serviceList.getService(n);
			service.byebye(bindAddr);
		}

		DeviceList childDeviceList = getDeviceList();
		int childDeviceCnt = childDeviceList.size();
		for (int n=0; n<childDeviceCnt; n++) {
			Device childDevice = childDeviceList.getDevice(n);
			childDevice.byebye(bindAddr);
		}
	}

	public void byebye()
	{
		int nHostAddrs = HostInterface.getNHostAddresses();
		for (int n=0; n<nHostAddrs; n++) {
			String bindAddr = HostInterface.getHostAddress(n);
			if (bindAddr == null || bindAddr.length() <= 0)
				continue;
			byebye(bindAddr);
		}
	}

	////////////////////////////////////////////////
	//	Search
	////////////////////////////////////////////////

	public void deviceSearchReceived(SSDPPacket ssdpPacket)
	{
		String remoteAddr = ssdpPacket.getRemoteAddress();
		int remotePort = ssdpPacket.getRemotePort();
		int mx = ssdpPacket.getMX();
	
		String localAddr = ssdpPacket.getLocalAddress();
		String rootDevLocation = getLocationURL(localAddr);
		String rootDevUSN = getUDN() + "::" + USN.ROOTDEVICE;
		
		SSDPSearchResponse ssdpRes = new SSDPSearchResponse();
		ssdpRes.setLeaseTime(getLeaseTime());
		ssdpRes.setDate(Calendar.getInstance());
		ssdpRes.setST(ST.ROOT_DEVICE);
		ssdpRes.setUSN(rootDevUSN);
		ssdpRes.setLocation(rootDevLocation);

		TimerUtil.waitRandom(mx * 1000);
		
		SSDPSearchResponseSocket ssdpResSock = new SSDPSearchResponseSocket();
		if (Debug.isOn() == true)
			ssdpRes.print();
		ssdpResSock.post(remoteAddr, remotePort, ssdpRes);
	}
	
	////////////////////////////////////////////////
	//	HTTP Server	
	////////////////////////////////////////////////

	public void setHTTPPort(int port)
	{
		getDeviceData().setHTTPPort(port);
	}
	
	public int getHTTPPort()
	{
		return getDeviceData().getHTTPPort();
	}

	public void httpRequestRecieved(HTTPRequest httpReq)
	{
		if (Debug.isOn() == true)
			httpReq.print();
	
		if (httpReq.isGetRequest() == true) {
			httpGetRequestRecieved(httpReq);
			return;
		}
		if (httpReq.isPostRequest() == true) {
			httpPostRequestRecieved(httpReq);
			return;
		}

		if (httpReq.isSubscribeRequest() == true || httpReq.isUnsubscribeRequest() == true) {
			SubscriptionRequest subReq = new SubscriptionRequest(httpReq);
			deviceEventSubscriptionRecieved(subReq);
			return;
		}

		httpReq.returnBadRequest();
	}

	private synchronized byte[] getDescriptionData(String host)
	{
		updateURLBase(host);
		Node rootNode = getRootNode();
		if (rootNode == null)
			return new byte[0];
		return rootNode.toString().getBytes();
	}
	
	private void httpGetRequestRecieved(HTTPRequest httpReq)
	{
		String uri = httpReq.getURI();
		Debug.message("httpGetRequestRecieved = " + uri);
		if (uri == null) {
			httpReq.returnBadRequest();
			return;
		}
					
		String rootPath = getDescriptionFilePath();
		
		byte fileByte[] = new byte[0];
		try {
			if (isDescriptionURI(uri) == true) {
				String localAddr = httpReq.getLocalAddress();
				fileByte = getDescriptionData(localAddr);
			}
			else {
				File docFile = new File(rootPath, uri);
				if (docFile.exists() == false) {
					httpReq.returnBadRequest();
					return;
				}
				fileByte = FileUtil.load(docFile); 
			}
		}
		catch (Exception e) {
			httpReq.returnBadRequest();
			return;
		}

		HTTPResponse httpRes = new HTTPResponse();
		if (FileUtil.isXMLFileName(uri) == true)
			httpRes.setContentType(XML.CONTENT_TYPE);
		httpRes.setStatusCode(HTTPStatus.OK);
		httpRes.setContent(fileByte);

		httpReq.post(httpRes);
	}

	private void httpPostRequestRecieved(HTTPRequest httpReq)
	{
		if (httpReq.isSOAPAction() == true) {
			//SOAPRequest soapReq = new SOAPRequest(httpReq);
			soapActionRecieved(httpReq);
		}
	}

	////////////////////////////////////////////////
	//	SOAP
	////////////////////////////////////////////////

	private void soapBadActionRecieved(HTTPRequest soapReq)
	{
		SOAPResponse soapRes = new SOAPResponse();
		soapRes.setStatusCode(HTTPStatus.BAD_REQUEST);
		soapReq.post(soapRes);
	}

	private void soapActionRecieved(HTTPRequest soapReq)
	{
		String uri = soapReq.getURI();
		Service ctlService = getServiceByControlURL(uri);
		if (ctlService != null)  {
			ActionRequest crlReq = new ActionRequest(soapReq);
			deviceControlRequestRecieved(crlReq, ctlService);
			return;
		}
		soapBadActionRecieved(soapReq);
	}

	////////////////////////////////////////////////
	//	controlAction
	////////////////////////////////////////////////

	private void deviceControlRequestRecieved(ControlRequest ctlReq, Service service)
	{
		if (ctlReq.isQueryControl() == true)
			deviceQueryControlRecieved(new QueryRequest(ctlReq), service);
		else
			deviceActionControlRecieved(new ActionRequest(ctlReq), service);
	}

	private void invalidActionControlRecieved(ControlRequest ctlReq)
	{
		ControlResponse actRes = new ActionResponse();
		actRes.setFaultResponse(UPnPStatus.INVALID_ACTION);
		ctlReq.post(actRes);
	}

	private void deviceActionControlRecieved(ActionRequest ctlReq, Service service)
	{
		if (Debug.isOn() == true)
			ctlReq.print();
			
		String actionName = ctlReq.getActionName();
		Action action = service.getAction(actionName);
		if (action == null) {
			invalidActionControlRecieved(ctlReq);
			return;
		}
		ArgumentList actionArgList = action.getArgumentList();
		ArgumentList reqArgList = ctlReq.getArgumentList();
		actionArgList.set(reqArgList);
		if (action.performActionListener(ctlReq) == false)
			invalidActionControlRecieved(ctlReq);
	}

	private void deviceQueryControlRecieved(QueryRequest ctlReq, Service service)
	{
		if (Debug.isOn() == true)
			ctlReq.print();
		String varName = ctlReq.getVarName();
		if (service.hasStateVariable(varName) == false) {
			invalidActionControlRecieved(ctlReq);
			return;
		}
		StateVariable stateVar = getStateVariable(varName);
		if (stateVar.performQueryListener(ctlReq) == false)
			invalidActionControlRecieved(ctlReq);
	}

	////////////////////////////////////////////////
	//	eventSubscribe
	////////////////////////////////////////////////

	private void upnpBadSubscriptionRecieved(SubscriptionRequest subReq, int code)
	{
		SubscriptionResponse subRes = new SubscriptionResponse();
		subRes.setErrorResponce(code);
		subReq.post(subRes);
	}

	private void deviceEventSubscriptionRecieved(SubscriptionRequest subReq)
	{
		String uri = subReq.getURI();
		Service service = getServiceByEventSubURL(uri);
		if (service == null) {
			subReq.returnBadRequest();
			return;
		}
		if (subReq.hasCallback() == false && subReq.hasSID() == false) {
			upnpBadSubscriptionRecieved(subReq, HTTPStatus.PRECONDITION_FAILED);
			return;
		}

		// UNSUBSCRIBE
		if (subReq.isUnsubscribeRequest() == true) {
			deviceEventUnsubscriptionRecieved(service, subReq);
			return;
		}

		// SUBSCRIBE (NEW)
		if (subReq.hasCallback() == true) {
			deviceEventNewSubscriptionRecieved(service, subReq);
			return;
		}
		
		// SUBSCRIBE (RENEW)
		if (subReq.hasSID() == true) {
			deviceEventRenewSubscriptionRecieved(service, subReq);
			return;
		}
		
		upnpBadSubscriptionRecieved(subReq, HTTPStatus.PRECONDITION_FAILED);
	}

	private void deviceEventNewSubscriptionRecieved(Service service, SubscriptionRequest subReq)
	{
		String callback = subReq.getCallback();
		try {
			URL url = new URL(callback);
		}
		catch (Exception e) {
			upnpBadSubscriptionRecieved(subReq, HTTPStatus.PRECONDITION_FAILED);
			return;
		}

		long timeOut = subReq.getTimeout();
		String sid = Subscription.createSID();
			
		Subscriber sub = new Subscriber();
		sub.setDeliveryURL(callback);
		sub.setTimeOut(timeOut);
		sub.setSID(sid);
		service.addSubscriber(sub);
			
		SubscriptionResponse subRes = new SubscriptionResponse();
		subRes.setStatusCode(HTTPStatus.OK);
		subRes.setSID(sid);
		subRes.setTimeout(timeOut);
		if (Debug.isOn() == true)
			subRes.print();
		subReq.post(subRes);

		service.notifyAllStateVariables();
	}

	private void deviceEventRenewSubscriptionRecieved(Service service, SubscriptionRequest subReq)
	{
		String sid = subReq.getSID();
		Subscriber sub = service.getSubscriber(sid);

		if (sub == null) {
			upnpBadSubscriptionRecieved(subReq, HTTPStatus.PRECONDITION_FAILED);
			return;
		}

		long timeOut = subReq.getTimeout();
		sub.setTimeOut(timeOut);
		sub.renew();
				
		SubscriptionResponse subRes = new SubscriptionResponse();
		subRes.setStatusCode(HTTPStatus.OK);
		subRes.setSID(sid);
		subRes.setTimeout(timeOut);
		subReq.post(subRes);
	}		

	private void deviceEventUnsubscriptionRecieved(Service service, SubscriptionRequest subReq)
	{
		String sid = subReq.getSID();
		Subscriber sub = service.getSubscriber(sid);

		if (sub == null) {
			upnpBadSubscriptionRecieved(subReq, HTTPStatus.PRECONDITION_FAILED);
			return;
		}

		service.removeSubscriber(sub);
						
		SubscriptionResponse subRes = new SubscriptionResponse();
		subRes.setStatusCode(HTTPStatus.OK);
		subReq.post(subRes);
	}		
	
	////////////////////////////////////////////////
	//	Thread	
	////////////////////////////////////////////////

	private HTTPServerList getHTTPServerList() 
	{
		return getDeviceData().getHTTPServerList();
	}

	private SSDPSearchSocketList getSSDPSearchSocketList() 
	{
		return getDeviceData().getSSDPSearchSocketList();
	}

	private void setAdvertiser(Advertiser adv) 
	{
		getDeviceData().setAdvertiser(adv);
	}
	
	private Advertiser getAdvertiser() 
	{
		return getDeviceData().getAdvertiser();
	}

	public boolean start()
	{
		stop(false);
		
		////////////////////////////////////////
		// HTTP Server
		////////////////////////////////////////
		
		int retryCnt = 0;
		int bindPort = getHTTPPort();
		HTTPServerList httpServerList = getHTTPServerList();
		while (httpServerList.open(bindPort) == false) {
			retryCnt++;
			if (UPnP.SERVER_RETRY_COUNT < retryCnt)
				return false;
			setHTTPPort(bindPort + 1);
			bindPort = getHTTPPort();
		}
		httpServerList.addRequestListener(this);
		httpServerList.start();

		////////////////////////////////////////
		// SSDP Seach Socket
		////////////////////////////////////////
		
		SSDPSearchSocketList ssdpSearchSockList = getSSDPSearchSocketList();
		if (ssdpSearchSockList.open() == false)
			return false;
		ssdpSearchSockList.addSearchListener(this);
		ssdpSearchSockList.start();

		////////////////////////////////////////
		// Announce
		////////////////////////////////////////
		
		announce();
		
		////////////////////////////////////////
		// Advertiser
		////////////////////////////////////////

		Advertiser adv = new Advertiser(this);
		setAdvertiser(adv);
		adv.start();
		
		return true;
	}

	private boolean stop(boolean doByeBye)
	{
		if (doByeBye == true)
			byebye();
		
		HTTPServerList httpServerList = getHTTPServerList();
		httpServerList.stop();
		httpServerList.close();
		httpServerList.clear();
		
		SSDPSearchSocketList ssdpSearchSockList = getSSDPSearchSocketList();
		ssdpSearchSockList.stop();
		ssdpSearchSockList.close();
		ssdpSearchSockList.clear();
		
		Advertiser adv = getAdvertiser();
		if (adv != null) {
			adv.stop();
			setAdvertiser(null);
		}

		return true;
	}
	
	public boolean stop()
	{
		return stop(true);
	}

	////////////////////////////////////////////////
	// Interface Address
	////////////////////////////////////////////////
	
	public String getInterfaceAddress() 
	{
		SSDPPacket ssdpPacket = getSSDPPacket();
		if (ssdpPacket == null)
			return "";
		return ssdpPacket.getLocalAddress();
	}

	////////////////////////////////////////////////
	// Acion/QueryListener
	////////////////////////////////////////////////
	
	public void setActionListener(ActionListener listener)
	{
		ServiceList serviceList = getServiceList();
		int nServices = serviceList.size();
		for (int n=0; n<nServices; n++) {
			Service service = serviceList.getService(n);
			service.setActionListener(listener);
		}
	}

	public void setQueryListener(QueryListener listener)
	{
		ServiceList serviceList = getServiceList();
		int nServices = serviceList.size();
		for (int n=0; n<nServices; n++) {
			Service service = serviceList.getService(n);
			service.setQueryListener(listener);
		}
	}

	////////////////////////////////////////////////
	//	output
	////////////////////////////////////////////////

/*
	public void output(PrintWriter ps) 
	{
		ps.println("deviceType = " + getDeviceType());
		ps.println("freindlyName = " + getFriendlyName());
		ps.println("presentationURL = " + getPresentationURL());

		DeviceList devList = getDeviceList();
		ps.println("devList = " + devList.size());
		
		ServiceList serviceList = getServiceList();
		ps.println("serviceList = " + serviceList.size());

		IconList iconList = getIconList();
		ps.println("iconList = " + iconList.size());
	}

	public void print()
	{
		PrintWriter pr = new PrintWriter(System.out);
		output(pr);
		pr.flush();
	}
*/

}

